<?php
// index.php - IPTV Player with file/URL loader + improved proxy, HTTPS/HTTP auto fix
?>
<!DOCTYPE html>
<html lang="pl">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>IPTV Player</title>
<style>
body {
    background: #1a1a1a;
    color: #fff;
    font-family: "Segoe UI", sans-serif;
    margin: 0;
    padding: 0;
    display: flex;
    flex-direction: column;
    height: 100vh;
}
header {
    background: #222;
    padding: 10px 16px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}
header h1 {
    font-size: 1.2rem;
    margin: 0;
}
.controls-top {
    display: flex;
    gap: 8px;
    align-items: center;
}
button, input[type="file"], input[type="text"] {
    background: #333;
    color: #fff;
    border: none;
    padding: 6px 10px;
    border-radius: 6px;
}
button:hover {
    background: #555;
    cursor: pointer;
}
input[type="text"] {
    width: 200px;
}
#proxyToggle {
    background: #444;
}
main {
    display: flex;
    flex: 1;
    overflow: hidden;
}
aside {
    width: 280px;
    background: #111;
    overflow-y: auto;
    border-right: 2px solid #222;
}
#search {
    width: 100%;
    padding: 8px;
    border: none;
    outline: none;
    background: #222;
    color: #fff;
}
.item {
    padding: 10px;
    border-bottom: 1px solid #222;
    cursor: pointer;
}
.item:hover {
    background: #333;
}
.title {
    font-weight: bold;
    font-size: 0.95rem;
}
.meta {
    font-size: 0.8rem;
    color: #aaa;
}
section {
    flex: 1;
    display: flex;
    flex-direction: column;
}
.playerCard {
    background: #000;
    flex: 1;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
}
video {
    width: 100%;
    height: 100%;
    background: #000;
}
.controls {
    background: #222;
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 6px 12px;
}
.now {
    font-size: 0.9rem;
    color: #ddd;
}
.now span {
    display: block;
    font-size: 0.75rem;
    color: #aaa;
    word-break: break-all;
}
</style>
</head>
<body>
<header>
    <h1>IPTV Player</h1>
    <div class="controls-top">
        <input type="file" id="fileInput" accept=".m3u">
        <input type="text" id="urlInput" placeholder="Wklej URL .m3u">
        <button id="loadUrlBtn">Załaduj</button>
        <button id="proxyToggle">Proxy: on</button>
    </div>
</header>

<main>
    <aside>
        <input type="text" id="search" placeholder="Szukaj kanału...">
        <div id="channelsContainer"></div>
    </aside>
    <section>
        <div class="playerCard">
            <video id="player" controls playsinline></video>
        </div>
        <div class="controls">
            <div class="now">
                <strong id="nowName">Brak</strong>
                <span id="nowUrl">–</span>
            </div>
            <div>
                <button id="prevBtn">⏮️</button>
                <button id="nextBtn">⏭️</button>
            </div>
        </div>
    </section>
</main>

<script src="https://cdn.jsdelivr.net/npm/hls.js@1.5.0/dist/hls.min.js"></script>
<script>
let channels = [];
let current = -1;
let useProxy = true;

const player = document.getElementById('player');
const nowName = document.getElementById('nowName');
const nowUrl = document.getElementById('nowUrl');
const proxyBtn = document.getElementById('proxyToggle');
const container = document.getElementById('channelsContainer');

/**
 * Tworzy poprawny URL proxy.php w tym samym katalogu i protokole (http/https)
 */
function proxyUrl(url){
    if (!useProxy) return url;
    const base = window.location.origin + window.location.pathname.replace(/\/[^\/]*$/, '/');
    return base + 'proxy.php?url=' + encodeURIComponent(url);
}

/**
 * Jeśli strona działa przez HTTPS, a link jest HTTP — wymusza użycie proxy
 */
function ensureProxyForMixedContent(url){
    const pageIsHttps = window.location.protocol === 'https:';
    const isHttpLink = url.startsWith('http://');
    if (pageIsHttps && isHttpLink) {
        useProxy = true;
        proxyBtn.textContent = 'Proxy: on';
    }
}

function isHls(url){ return /\.m3u8($|\?)/i.test(url); }
function isTs(url){ return /\.ts($|\?)/i.test(url); }

function clearPlayer(){
    try { player.pause(); } catch(e){}
    try { player.removeAttribute('src'); player.load(); } catch(e){}
    if (window.hlsInstance) { try { window.hlsInstance.destroy(); } catch(e){} window.hlsInstance = null; }
}

function showMessage(msg, type){
    let el = document.getElementById('playerMsg');
    if (!el) {
        el = document.createElement('div');
        el.id = 'playerMsg';
        el.style.marginTop = '8px';
        el.style.fontSize = '0.9rem';
        document.querySelector('.playerCard').appendChild(el);
    }
    el.textContent = msg;
    el.style.color = (type === 'error') ? '#ff8a8a' : '#ffdca8';
}

function renderChannels(){
    container.innerHTML = '';
    channels.forEach((ch,i)=>{
        const div = document.createElement('div');
        div.className = 'item';
        div.dataset.idx = i;
        div.innerHTML = `<div class="title">${ch.name}</div><div class="meta">${ch.meta||''}</div>`;
        div.onclick = ()=>playChannel(i);
        container.appendChild(div);
    });
}

function parseM3U(content){
    const lines = content.split(/\r?\n/);
    let list = [];
    let current = {};
    for (let line of lines){
        if (line.startsWith('#EXTINF:')){
            const info = line.substring(8).split(',',2);
            current = {meta: info[0], name: info[1]||info[0]};
        } else if (line.trim() && !line.startsWith('#')){
            current.url = line.trim();
            list.push(current);
            current = {};
        }
    }
    channels = list;
    renderChannels();
    if (channels.length>0) playChannel(0);
}

function playChannel(idx){
    if (!channels[idx]) return;
    current = idx;
    const ch = channels[idx];
    nowName.textContent = ch.name;
    nowUrl.textContent = ch.url;
    let rawUrl = ch.url;

    // automatyczne włączenie proxy, jeśli trzeba
    ensureProxyForMixedContent(rawUrl);
    let url = proxyUrl(rawUrl);

    clearPlayer();
    showMessage('Ładowanie...', 'info');

    if (isHls(rawUrl)){
        if (Hls.isSupported()) {
            const hls = new Hls();
            window.hlsInstance = hls;
            hls.loadSource(url);
            hls.attachMedia(player);
            hls.on(Hls.Events.MANIFEST_PARSED, ()=>{player.play().catch(()=>{});});
            hls.on(Hls.Events.ERROR, (e,d)=>showMessage('Błąd HLS: '+(d.type||'unknown'),'error'));
        } else if (player.canPlayType('application/vnd.apple.mpegurl')){
            player.src = url;
            player.addEventListener('loadedmetadata', ()=>player.play().catch(()=>{}), {once:true});
        } else showMessage('Brak wsparcia HLS.','error');
        return;
    }

    if (isTs(rawUrl)){
        player.src = url;
        player.addEventListener('error', ()=>showMessage('Brak obsługi .ts w tej przeglądarce','error'), {once:true});
        player.play().catch(()=>{});
        return;
    }

    player.src = url;
    player.addEventListener('error', ()=>showMessage('Błąd odtwarzania','error'), {once:true});
    player.play().catch(()=>{});
}

document.getElementById('search').addEventListener('input', function(){
    const q = this.value.toLowerCase();
    document.querySelectorAll('.item').forEach(it=>{
        const txt = it.textContent.toLowerCase();
        it.style.display = txt.includes(q) ? '' : 'none';
    });
});

document.getElementById('prevBtn').onclick = ()=>{ if (channels.length) playChannel((current<=0)?channels.length-1:current-1); };
document.getElementById('nextBtn').onclick = ()=>{ if (channels.length) playChannel((current>=channels.length-1)?0:current+1); };

proxyBtn.onclick = ()=>{
    useProxy = !useProxy;
    proxyBtn.textContent = 'Proxy: ' + (useProxy ? 'on' : 'off');
    if (current >= 0) playChannel(current);
};

document.getElementById('fileInput').addEventListener('change', function(e){
    const file = e.target.files[0];
    if (!file) return;
    const reader = new FileReader();
    reader.onload = ev => parseM3U(ev.target.result);
    reader.readAsText(file);
});

document.getElementById('loadUrlBtn').onclick = ()=>{
    const url = document.getElementById('urlInput').value.trim();
    if (!url) return alert('Podaj URL playlisty .m3u');
    ensureProxyForMixedContent(url);
    fetch(proxyUrl(url))
        .then(r=>r.text())
        .then(txt=>parseM3U(txt))
        .catch(()=>alert('Nie udało się wczytać playlisty'));
};
</script>
</body>
</html>
